using System;
using System.Linq;
using System.Data;
using System.Data.SqlClient;
using System.Reflection;
using System.Collections;
using gov.va.med.vbecs.ExceptionManagement;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using STOREDPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;
using gov.va.med.vbecs.Common;
using System.Collections.Generic;

namespace gov.va.med.vbecs.DAL
{

    #region Header

    ///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
    ///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
    ///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
    ///<Developers>
    ///	<Developer>Doug Forst</Developer>
    ///</Developers>
    ///<SiteName></SiteName>
    ///<CreationDate>12/18/2015</CreationDate>
    ///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
    ///<summary>
    ///	Data access component providing database read/write functionality for
    ///	pending blood unit tests received from an automated instrument.
    ///</summary>

    #endregion
    public class PendingBloodUnitTest
    {
        ///<Developers>
        ///	<Developer>Doug Forst</Developer>
        ///</Developers>
        ///<SiteName></SiteName>
        ///<CreationDate>02/05/2016</CreationDate>
        ///<summary>
        /// Gets empty DataTable for updating the PendingBloodUnit table
        ///</summary>
        ///<returns>DataTable containing columns for updating PendingBloodUnit tests</returns>
        public static DataTable GetEmptyPendingBloodUnitTestTableSchemaForUpdate()
        {
            DataTable dtPendingBloodUnitTest = new DataTable(TABLE.PendingBloodUnitTest.TableName);
            dtPendingBloodUnitTest.Columns.Add(TABLE.PendingBloodUnitTest.PendingBloodUnitTestId, typeof(int));
            dtPendingBloodUnitTest.Columns.Add(TABLE.PendingBloodUnitTest.BloodUnitTestGuid, typeof(Guid));
            dtPendingBloodUnitTest.Columns.Add(TABLE.PendingBloodUnitTest.PendingTestStatusId, typeof(byte));
            dtPendingBloodUnitTest.Columns.Add(TABLE.PendingBloodUnitTest.RejectionComment, typeof(string));
            dtPendingBloodUnitTest.Columns.Add(TABLE.PendingBloodUnitTest.LastUpdateUser, typeof(string));
            dtPendingBloodUnitTest.Columns.Add(TABLE.PendingBloodUnitTest.LastUpdateFunctionId, typeof(int));
            dtPendingBloodUnitTest.Columns.Add(TABLE.PendingBloodUnitTest.RowVersion, typeof(byte[]));
            return dtPendingBloodUnitTest;
        }

        ///<Developers>
        ///	<Developer>Doug Forst</Developer>
        ///</Developers>
        ///<SiteName></SiteName>
        ///<CreationDate>12/18/2015</CreationDate>
        ///<summary>
        /// Gets pending blood unit test results data for a pending blood units with a given division code and pending test status id
        ///</summary>
        ///<param name="divisionCode">Division Code</param>
        ///<param name="pendingTestStatus">Pending Test Status</param>
        ///<returns>DataTable containing pending blood unit test results</returns>
        public static DataSet GetPendingBloodUnitTestResults(string divisionCode, PendingTestStatus pendingTestStatus)
        {
            SqlParameter[] prms =
				{
                    new SqlParameter(STOREDPROC.GetPendingBloodUnitTests.DivisionCode, System.Data.SqlDbType.VarChar),
                    new SqlParameter(STOREDPROC.GetPendingBloodUnitTests.PendingTestStatusId, System.Data.SqlDbType.TinyInt),
				};

            prms[0].Value = divisionCode;
            prms[1].Value = (byte)pendingTestStatus;

            return Common.StoredProcedure.GetData(STOREDPROC.GetPendingBloodUnitTests.StoredProcName, prms);
        }

        ///<Developers>
        ///	<Developer>Doug Forst</Developer>
        ///</Developers>
        ///<SiteName></SiteName>
        ///<CreationDate>12/29/2015</CreationDate>
        ///<summary>
        /// Gets count of pending blood unit tests for a given division code and pending test status id
        ///</summary>
        ///<param name="divisionCode">Division Code</param>
        ///<param name="pendingTestStatus">Pending Test Status</param>
        ///<returns>Count of pending blood unit tests</returns>
        public static int GetCount(string divisionCode, PendingTestStatus pendingTestStatus)
        {
            SqlParameter[] prms =
				{
                    new SqlParameter(STOREDPROC.GetPendingBloodUnitTestsCount.divisioncode, System.Data.SqlDbType.VarChar),
                    new SqlParameter(STOREDPROC.GetPendingBloodUnitTestsCount.pendingteststatusid, System.Data.SqlDbType.TinyInt),
				};

            prms[0].Value = divisionCode;
            prms[1].Value = (byte)pendingTestStatus;

            return int.Parse(StoredProcedure.GetSingleDataRow(STOREDPROC.GetPendingBloodUnitTestsCount.StoredProcName, prms)[0].ToString());
        }


        ///<Developers>
        ///	<Developer>Doug Forst</Developer>
        ///</Developers>
        ///<SiteName></SiteName>
        ///<CreationDate>01/04/2016</CreationDate>
        /// <summary>
        /// Update pending blood unit tests
        /// </summary>
        /// <param name="dtUpdatePendingBloodUnitTest"></param>
        public static bool Update(DataTable dtUpdatePendingBloodUnitTest)
        {
            return (new StoredProcedure().TransactionalGetValue(STOREDPROC.UpdatePendingBloodUnitTest.StoredProcName, dtUpdatePendingBloodUnitTest) == 0);
        }

        private static SqlParameter BuildSqlParameter(string parameterName, SqlDbType sqlDbType, object value)
        {
            var param = new SqlParameter(parameterName, sqlDbType);
            param.Value = value;
            return param;
        }

        ///<Developers>
        ///	<Developer>Doug Forst</Developer>
        ///</Developers>
        ///<SiteName></SiteName>
        ///<CreationDate>01/06/2016</CreationDate>
        ///<summary>
        /// Gets valid AboRhConfirmation blood units for the given division code and unit id
        /// Defect 286187 (rename spGetBloodUnitByUnitId to spGetBloodUnitsByUnitId)
        ///</summary>
        ///<param name="divisionCode">Division Code</param>
        ///<param name="bloodUnitId">Pending Test Status</param>
        ///<returns>Count of pending blood unit tests</returns>
        public static DataTable GetBloodUnitsByUnitId(string divisionCode, string bloodUnitId)
        {
            IList<SqlParameter> prms = new List<SqlParameter>()
				{
                    BuildSqlParameter(STOREDPROC.GetBloodUnitsByUnitId.divisionCode, System.Data.SqlDbType.VarChar, divisionCode),
                    BuildSqlParameter(STOREDPROC.GetBloodUnitsByUnitId.unitId, System.Data.SqlDbType.VarChar, bloodUnitId),
				};

            return StoredProcedure.GetDataTable(STOREDPROC.GetBloodUnitsByUnitId.StoredProcName, prms.ToArray());
        }

        ///<Developers>
        ///	<Developer>Doug Forst</Developer>
        ///</Developers>
        ///<SiteName></SiteName>
        ///<CreationDate>01/06/2016</CreationDate>
        ///<summary>
        /// Gets valid XM blood units for the given division code and unit id
        /// Defect 282082
        ///</summary>
        ///<param name="divisionCode">Division Code</param>
        ///<param name="bloodUnitId">Blood Unit Id</param>
        ///<param name="specimenUid">Specimen Uid</param>
        ///<returns>Blood units available for the given Blood Unit Id and SpecimenUid</returns>
        public static DataTable GetXMBloodUnitsBySpecimenUIDAndUnitId(string divisionCode, string bloodUnitId, string specimenUid)
        {
            IList<SqlParameter> prms = new List<SqlParameter>()
				{
                    BuildSqlParameter(STOREDPROC.GetXMBloodUnitsBySpecimenUIDAndUnitID.divisionCode, System.Data.SqlDbType.VarChar, divisionCode),
                    BuildSqlParameter(STOREDPROC.GetXMBloodUnitsBySpecimenUIDAndUnitID.eyeReadableUnitId, System.Data.SqlDbType.VarChar, bloodUnitId),
                    BuildSqlParameter(STOREDPROC.GetXMBloodUnitsBySpecimenUIDAndUnitID.specimenuid, System.Data.SqlDbType.VarChar, specimenUid),
				};

            return StoredProcedure.GetDataTable(STOREDPROC.GetXMBloodUnitsBySpecimenUIDAndUnitID.StoredProcName, prms.ToArray());
        }

        ///<Developers>
        ///<Developer>Doug Forst</Developer>
        ///</Developers>
        ///<SiteName></SiteName>
        ///<CreationDate>02/04/2016</CreationDate>
        ///<summary>
        /// Gets pending specimen test results for a given patientSpecimenGuid, division code and pending test status id
        ///</summary>
        ///<param name="patientSpecimenGuid">SpecimenUid</param>
        ///<param name="divisionCode">Division Code</param>
        ///<param name="pendingTestStatus">Pending Test Status</param>
        ///<returns>DataTable containing XM pending blood unit test results</returns>
        public static DataSet GetPendingBloodUnitXMTests(Guid patientSpecimenGuid, string divisionCode, PendingTestStatus pendingTestStatus)
        {
            IList<SqlParameter> prms = new List<SqlParameter>()
				{
                    StoredProcedure.MakeParameter(STOREDPROC.GetPendingBloodUnitXMTests.PatientSpecimenGuid, patientSpecimenGuid),
                    StoredProcedure.MakeParameter(STOREDPROC.GetPendingBloodUnitXMTests.DivisionCode, divisionCode),
                    StoredProcedure.MakeParameter(STOREDPROC.GetPendingBloodUnitXMTests.PendingTestStatusId, (byte)pendingTestStatus),
				};

            return Common.StoredProcedure.GetData(STOREDPROC.GetPendingBloodUnitXMTests.StoredProcName, prms.ToArray());
        }
    }
}